import SwiftUI
import Girocard_Branding_SDK

struct ContentView: View {
    @State private var audioToggle = true
    @State private var vibrationToggle = true
    @State private var playingInView = false
    @State private var playingFullScreen = false
    @State private var buttonsDisabled = false
    let girocardBrandingService = GirocardBrandingServiceLogic()
    
    let girocardBrandingView = GirocardBrandingView()
    
    var body: some View {
        VStack {
            VStack(spacing: 1) {
                ToggleView(title: "Include Sound", isToggled: $audioToggle)
                ToggleView(title: "Use Vibration", isToggled: $vibrationToggle)
            }
            
            ButtonView(title: "PLAY", buttonAction: {
                playingInView = true
                play()
            }).padding(.vertical, 5).disabled(buttonsDisabled)
            
            ButtonView(title: "PLAY FULLSCREEN", buttonAction: {
                playingFullScreen = true
                play()
            }).padding(.vertical, 5).disabled(buttonsDisabled)
            
            if playingInView {
                GirocardBrandingSwitUIView(view: girocardBrandingView)
            }
            
            Spacer()
        }
        .fullScreenCover(isPresented: $playingFullScreen) {
            GirocardBrandingSwitUIView(view: girocardBrandingView)
                .ignoresSafeArea(.all)
        }
    }
    
    private func play() {
        buttonsDisabled = true
        let config = extractConfig()
        
        girocardBrandingService.play(in: girocardBrandingView, 
                                    configuration: config) {
            print("success")
            playingFullScreen = false
            playingInView = false
            buttonsDisabled = false
        } failure: { girocardError in
            print("failure with error: [\(girocardError.code)] \(girocardError.localizedDescription)")
            playingFullScreen = false
            playingInView = false
            buttonsDisabled = false
        }
    }
    
    private func extractConfig() -> GirocardBrandingConfiguration {
        return GirocardBrandingConfiguration(includeSound: audioToggle,
                                            useVibration: vibrationToggle)
    }
}

struct ToggleView: View {
    @Binding private var isToggled: Bool
    let title: String
    
    init(title: String, isToggled: Binding<Bool>) {
        self.title = title
        self._isToggled = isToggled
    }

    var body: some View {
        Toggle(title, isOn: $isToggled)
            .padding()
            .toggleStyle(SwitchToggleStyle(tint: Color(CustomColor.main.rawValue)))
    }
}

struct ButtonView: View {
    let buttonAction: () -> ()
    let title: String
    
    init(title: String, buttonAction: @escaping () -> ()) {
        self.title = title
        self.buttonAction = buttonAction
    }
    var body: some View {
        Button(action: {
            buttonAction()
        }) {
            Text(title)
                .padding()
                .frame(width: 350, height: 40)
                .background(Color(CustomColor.main.rawValue))
                .foregroundColor(.white)
                .cornerRadius(10)
        }
    }
}

#Preview {
    ContentView()
}

enum CustomColor: String {
    case main
}

