import AVFoundation
import UIKit

class AnimationServiceLogic: AnimationService {
    private var player = AVQueuePlayer()
    private var observers = [NSObjectProtocol]()
    private var animationIsPlaying = false
    private var success: (() -> ())?
    private var failure: ((GirocardError) -> ())?
    private var view: GirocardBrandingView?
    
    func play(view: GirocardBrandingView,
              configuration: GirocardBrandingConfiguration,
              animationStarted: @escaping () -> (),
              success: @escaping () -> (),
              failure: @escaping (GirocardError) -> ()) {
        guard !animationIsPlaying else {
            failure(ErrorComposer.girocardError(code: .animationAlreadyPlaying))
            return
        }
        
        guard let resources = try? extractAnimationURLs(from: configuration),
              !resources.isEmpty else {
            failure(ErrorComposer.girocardError(code: .resourcesMissing))
            return
        }
        
        self.view = view
        self.success = success
        self.failure = failure
        
        var items = [AVPlayerItem]()
        
        for resource in resources {
            let item = AVPlayerItem(url: URL(fileURLWithPath: resource))
            items.append(item)
        }
        
        player = AVQueuePlayer(items: items)
        player.isMuted = !configuration.includeSound
        registerNotifications(for: items.last!)
        
        let playerLayer = AVPlayerLayer(player: player)
        playerLayer.frame = view.bounds
        playerLayer.backgroundColor = UIColor.clear.cgColor
        
        view.addLayer(playerLayer)
        
        Logger.infoLog(message: "Playing Girocard branding animation.")
        
        animationIsPlaying = true
        player.play()
        animationStarted()
    }
    
    private func extractAnimationURLs(from configuration: GirocardBrandingConfiguration) throws -> [String] {
        var animationURLs = [String]()
        
        guard let animation = Bundle(for: GirocardBrandingView.self).path(forResource: PredefinedResources.animation, ofType: nil) else {
            throw ErrorComposer.girocardError(code: .resourcesMissing)
        }
        animationURLs.append(animation)
        return animationURLs
    }
    
    private func registerNotifications(for item: AVPlayerItem) {
        unregisterNotifications()
        observers = [
            NotificationCenter.default
                .addObserver(forName: .AVPlayerItemDidPlayToEndTime,
                             object: item,
                             queue: .main,
                             using: playerDidFinishPlaying(notification: )),
            NotificationCenter.default
                .addObserver(forName: .AVPlayerItemFailedToPlayToEndTime,
                             object: item,
                             queue: .main,
                             using: playerDidFailPlaying(notification: ))]
    }
    
    private func unregisterNotifications() {
        for observer in observers {
            NotificationCenter.default.removeObserver(observer)
        }
        observers = [NSObjectProtocol]()
    }
    
    @objc func playerDidFinishPlaying(notification: Notification) {
        Logger.infoLog(message: "Finished playing Girocard branding animation.")
        success?()
    }
    
    @objc func playerDidFailPlaying(notification: Notification) {
        Logger.infoLog(message: "Failed playing Girocard branding animation.")
        failure?(ErrorComposer.girocardError(code: .animationPlayerFailure))
    }
    
    func cleanUp() {
        Logger.infoLog(message: "Cleaning up animation service resources.")
        view?.removeLayers()
        unregisterNotifications()
        success = nil
        failure = nil
        animationIsPlaying = false
    }
}
