import Foundation
import CoreHaptics

class HapticFeedbackServiceLogic: HapticFeedbackService {
    private let hapticEngine: CHHapticEngine?
    
    init() {
        guard CHHapticEngine.capabilitiesForHardware().supportsHaptics else {
            Logger.errorLog(message: "Hardware doesn't support haptics.")
            self.hapticEngine = nil
            return
        }
        
        self.hapticEngine = try? CHHapticEngine()
    }
    
    func vibrate(with pattern: VibrationPattern) {
        guard let hapticEngine else {
            Logger.errorLog(message: "Failed to initialize haptic engine.")
            return
        }
        
        do {
            Logger.infoLog(message: "Haptic engine started.")
            try hapticEngine.start()
        } catch {
            Logger.errorLog(message: "Failed to start haptic engine.")
            return
        }
        
        let events = parse(pattern: pattern)
        
        do {
            let pattern = try CHHapticPattern(events: events, parameters: [])
            let player = try hapticEngine.makePlayer(with: pattern)
            Logger.infoLog(message: "Playing haptic feedback.")
            try player.start(atTime: 0)
        } catch {
            Logger.errorLog(message: "Failed to execute haptic pattern.")
        }
    }
    
    private func parse(pattern: VibrationPattern) -> [CHHapticEvent] {
        let millisecondsInSecond: Double = 1000
        let maxAmplitude: Float = 255
        
        let parsedTimings = pattern.timings.map { Double($0) / millisecondsInSecond }
        let parsedAmplitudes = pattern.amplitudes.map { Float($0) / maxAmplitude }
        
        if parsedTimings.count != parsedAmplitudes.count {
            Logger.errorLog(message: "Timings and Amplitudes count must be equal.")
            return []
        }
        
        var events = [CHHapticEvent]()
        var vibrationTime: Double = 0
        
        for i in 0..<parsedTimings.count {
            events.append(
                CHHapticEvent(eventType: .hapticContinuous,
                              parameters: [
                                CHHapticEventParameter(parameterID: .hapticIntensity, 
                                                       value: Float(parsedAmplitudes[i])),
                                CHHapticEventParameter(parameterID: .hapticSharpness,
                                                       value: Float(parsedAmplitudes[i]))
                              ],
                              relativeTime: vibrationTime,
                              duration: parsedTimings[i])
            )
            vibrationTime += parsedTimings[i]
        }
        
        return events
    }
    
    func cleanUp() {
        Logger.infoLog(message: "Cleaning up haptic feedback service resources.")
        hapticEngine?.stop()
    }
}
