import Foundation
import os.log

@objc
public enum LogLevel: Int, RawRepresentable {
    case debug
    case info
    case error
    case noLog
    
    public typealias RawValue = String
    
    public var rawValue: String {
        switch self {
        case .debug:
            return "debug"
        case .info:
            return "info"
        case .error:
            return "error"
        case .noLog:
            return "nolog"
        }
    }
    
    public init(rawValue: LogLevel.RawValue) {
        switch rawValue {
        case "debug":
            self = .debug
        case "info":
            self = .info
        case "error":
            self = .error
        case "nolog":
            self = .noLog
        default:
            self = .noLog
        }
    }
    
    internal func intValue() -> Int {
        switch self {
        case .debug:
            return 3
        case .info:
            return 2
        case .error:
            return 1
        case .noLog:
            return 0
        }
    }
}

class Logger {
    private static var logLevel = LogLevel.debug
    private static let date = Date()
    private static let subsystem =  PredefinedResources.domain
    private static let sdkLogger = OSLog(subsystem: subsystem, category: PredefinedResources.domain)
    
    internal static func log(to level: LogLevel) {
        self.logLevel = level
    }
    
    static func warningLog(message toLog: String) {
        infoLog(message: "Warning: \(toLog)")
    }
    
    static func infoLog(message toLog: String) {
        pass(message: toLog, toLevel: .info)
    }
    
    static func debugLog(message toLog: String) {
        pass(message: toLog, toLevel: .debug)
    }
    
    static func errorLog(message toLog: String) {
        pass(message: toLog, toLevel: .error)
    }
    
    private static func pass(message: String, toLevel: OSLogType) {
        switch toLevel {
        case OSLogType.debug :
            if logLevel.intValue() > 2 {
                log(message: message, toLevel: toLevel)
            }
        case OSLogType.info :
            if logLevel.intValue() > 1 {
                log(message: message, toLevel: toLevel)
            }
        case OSLogType.error :
            if logLevel.intValue() > 0 {
                log(message: message, toLevel: toLevel)
            }
        default:
            break
        }
    }
    
    private static func log(message: String, toLevel: OSLogType) {
        os_log("%{time_t}d - %{public}@ ", log: sdkLogger, type: toLevel, time_t(date.timeIntervalSince1970), message)
    }
}
